 
import numpy as np
from sklearn.linear_model import MultiTaskLasso, Lasso
import matplotlib.pyplot as plt
from matplotlib.font_manager import FontProperties

rng = np.random.RandomState(42)

# 通过使用正弦波形状的系数矩阵，构建多任务Lasso回归模型可以使用的训练数据集
print("开始构建训练数据集......")

# 1。定义训练数据集的形状属性
# n_samples：样本数量（训练数据)
# n_features：特征变量个数
# n_tasks：线性回归模型个数（任务数量）
# n_relevant_features：线性相关特征变量的个数
n_samples, n_features, n_tasks = 100, 30, 40
n_relevant_features = 5

# 2。定义并初始化回归系数矩阵（多任务Lasso回归）
# 2.1. 定义形状为(n_tasks, n_features)的回归系数矩阵
#    并初始化矩阵元素为0（数据类型为浮点数）。
coef = np.zeros((n_tasks, n_features))
# 2.2. 创建一个数据序列，包含n_tasks个数据元素，并使其值范围为[0, 2 * np.pi]
times = np.linspace(0, 2 * np.pi, n_tasks)  # times是一个Numpy数组
# 2.3. 初始化前n_relevant_features个特征变量
#    实际上就是使前n_relevant_features个特征变量线性相关
for k in range(n_relevant_features):
    coef[:, k] = np.sin((1.0 + rng.randn(1)) * times + 3 * rng.randn(1) )
#                          注：rng.randn(1)从标准正态分布中返回一个样本

# 3。构建训练数据集
# 构建训练数据集中的特征变量集合，即设计矩阵，其形状为(n_samples, n_features)
X = rng.randn(n_samples, n_features)
# 构建训练数据集中的目标变量集合，其形状为(n_samples, n_tasks)
# Y =      X*W        + ε
Y = np.dot(X, coef.T) + rng.randn(n_samples, n_tasks)
# --------- 训练数据集，包括特征变量集合和目标变量集合构建完毕
print( "训练数据集完毕：特征变量集的形状：%s，目标变量集的形状：%s。" % (X.shape, Y.shape) )
print( "其中前 %d 个特征变量线性相关。" % (n_relevant_features) )

# 4。根据Y矩阵中的每一个目标变量值序列，分别独立构建一个Lasso回归模型
# 并把回归系数存入 coef_lasso_
#coef_lasso_ = np.array([Lasso(alpha=0.5).fit(X, y).coef_ for y in Y.T])
# 下面处理虽然多行，但是容易理解。实现了上一句同样的功能。
coef_lasso_ = np.zeros((n_tasks, n_features))
iRow = 0
for y in Y.T:
    obj = Lasso(alpha=0.5).fit(X, y).coef_
    coef_lasso_[iRow][:] = obj
    iRow += 1

# 5。根据Y矩阵中的目标变量值序列，构建一个多任务Lasso回归模型
# 并把回归系数存入 coef_multi_task_lasso_
multi_task_lasso = MultiTaskLasso(alpha=1.0)
coef_multi_task_lasso_ = multi_task_lasso.fit(X, Y).coef_

# 6。预测方法predict()的使用
# 多任务Lasso回归模型的predict()方法的输入需要一个数组
x1 = X[0].reshape(1,-1)
y1 = multi_task_lasso.predict(x1)
print("预测方法predict()的使用")
print("输入数据：\n", x1, "\n")
print("预测数据：\n", y1, "\n")

# 7。预测方法score()的使用
r2_multi_task_lasso_ = multi_task_lasso.score(X, Y)
print("拟合优度R2", r2_multi_task_lasso_)

# 8。------------绘制图形，可视化多个模型的回归系数态势------------
# 通过这种方式可以局部设置字体（支持中文），不影响绘图其他部分
font = FontProperties(fname='C:\\Windows\\Fonts\\SimHei.ttf')  # , size=16

fig = plt.figure(figsize=(8, 5))
plt.subplot(1, 2, 1)
# 以图片的方式表示系数矩阵
plt.spy(coef_lasso_)
plt.xlabel('特征变量', fontproperties=font)
plt.ylabel('任务(模型)', fontproperties=font)
plt.text(10, 5, 'Lasso')

plt.subplot(1, 2, 2)
plt.spy(coef_multi_task_lasso_)
plt.xlabel('特征变量', fontproperties=font)
plt.ylabel('任务(模型)', fontproperties=font)
plt.text(10, 5, 'MultiTaskLasso')
# fig.suptitle('Coefficient non-zero location')
fig.suptitle('非零回归系数的位置', fontproperties=font)

feature_to_plot = 0
plt.figure()
plt.plot(coef[:, feature_to_plot],               color='seagreen',
         linewidth=2, linestyle='-',  label='观测值')
plt.plot(coef_multi_task_lasso_[:, feature_to_plot], color='gold',
         linewidth=2, linestyle=':',  label='MultiTaskLasso')
plt.plot(coef_lasso_[:, feature_to_plot],  color='cornflowerblue',
         linewidth=2, linestyle='--', label='Lasso' )
plt.legend(loc='upper center', prop = font)
plt.axis('tight')
plt.ylim([-1.1, 1.1])
plt.show()
