 
# 根据https://sebastianraschka.com/Articles/2014_kernel_pca.html改编 
import numpy as np
from scipy.linalg import eigh
from scipy.spatial.distance import pdist, squareform
from sklearn.datasets import make_moons
import matplotlib.pyplot as plt
from matplotlib.font_manager import FontProperties

def stepwise_kpca(X, gamma, n_components):
  """
  基于RBF的核主成分分析KPCA
  参数：
      X：一个Numpy数组，表示输入（原始）数据集
      gamma：RBF核函数的参数
      n_components：选择的主成分数量
  返回值：
      前k个最大特征值（lambdas）及其对应的特征向量（alphas）
   """
  # 计算输入数据集中两两点的平方欧几里德距离
  sq_dists = pdist(X, 'sqeuclidean')

  # 转换sq_dists为一个对称方阵
  mat_sq_dists = squareform(sq_dists)

  # 计算方阵的核矩阵
  K = np.exp(-gamma * mat_sq_dists)

  # 居中对称核矩阵
  N = K.shape[0]
  one_n = np.ones((N,N)) / N
  K_norm = K - one_n.dot(K) - K.dot(one_n) + one_n.dot(K).dot(one_n)

  # 获取按特征值倒排序的排列及其对应的特征向量
  eigvals, eigvecs = eigh(K_norm)

  # 获取前n_components个特征值及其对应的特征向量
  lambdas = [eigvals[-i] for i in range(1,n_components+1)]
  alphas = np.column_stack( list((eigvecs[:,-i] for i in list(range(1,n_components+1)))) )
  
  return lambdas, alphas
# end of stepwise_kpca()



# 生成两个交错半圆的数据集（两个类别），用于投影到一个1维子空间中（使用RBF核函数）
X, y = make_moons(n_samples=100, random_state=123)

# 首先绘制生成的数据集：散点图
# 设置绘图区域的尺寸，宽15，高8。单位inch
fig = plt.figure(figsize=(15,8))
fig.canvas.manager.set_window_title("核主成分分析KPCA")  # Matplotlib >= 3.4
#fig.canvas.set_window_title("核主成分分析KPCA")  # Matplotlib < 3.4
font = FontProperties(fname="C:\\Windows\\Fonts\\SimHei.ttf")  # , size=16

# 一行，两列。共两个图形：一个原始数据，一个转换后的数据
# 绘制原始数据的图形
plt.subplot(1, 2, 1)
plt.scatter(X[y==0, 0], X[y==0, 1], color="red", alpha=0.5)
plt.scatter(X[y==1, 0], X[y==1, 1], color="blue", alpha=0.5)

plt.title("非线性 2D 数据集", font=font)
plt.ylabel("y 轴", font=font)
plt.xlabel("x 轴", font=font)

#plt.show()


def project_x(x_new, X, gamma, lambdas, alphas):
  """
  单个点映射计算
  参数：
      x_new：一个新的数据样本
      X：原始数据集
      gamma：RBF核函数的参数
      lambdas, alphas：计算获取的特征值和特征向量
  返回值：
      映射到子空间的新数据值
   """
  pair_dist = np.array([np.sum((x_new-row)**2) for row in X])
  k = np.exp(-gamma * pair_dist)

  # 返回矩阵点积
  return k.dot(alphas / lambdas)
# end of project_x()


# 调用函数stepwise_kpca()
lambdas, alphas = stepwise_kpca(X, gamma=15, n_components=1)

# 随机去一个点，验证方法的正确性：
# 假设 X[25] 为一个新数据点，把它映射到新的子空间中（1维）
x_new  = X[25]
X_proj = alphas[25] # original projection

# x_new的映射
x_reproj = project_x(x_new, X, gamma=15, lambdas=lambdas, alphas=alphas)


# 绘制新子空间下的数据集图形：散点图
plt.subplot(1, 2, 2)
plt.scatter(alphas[y==0, 0], np.zeros((50)), color="red", alpha=0.5)
plt.scatter(alphas[y==1, 0], np.zeros((50)), color="blue", alpha=0.5)

# 随机选定的点
plt.scatter(X_proj,   0, color="black", label="数据样本X[25]的原始投影", marker="^", s=100)
plt.scatter(x_reproj, 0, color="green", label="X[25]的重新映射点", marker="x", s=200)
plt.legend(scatterpoints=1, prop=font)

plt.show()
 
