 
import numpy as np
from sklearn.datasets import load_iris
from sklearn.model_selection import train_test_split
from sklearn.ensemble import VotingClassifier
from sklearn.tree import DecisionTreeClassifier
from sklearn.neural_network import MLPClassifier
from sklearn.neighbors import KNeighborsClassifier
from matplotlib.font_manager import FontProperties
import matplotlib.pyplot as plt

#0 随机数种子
seed = 10

#1 导入鸢尾花数据集，并分离特征变量和目标变量
irisData = load_iris()
X0 = irisData.data
y0 = irisData.target

#2 本例子中，仅考虑前2个特征变量，以便能够绘制样本点以及分类器的决策边界
X = X0[:,0:2]
y = y0

#3 把数据集分割为训练数据集和测试数据集
X_train, X_test, y_train, y_test = train_test_split(X, y, test_size=0.34, stratify=y, random_state=seed)

#4 声明一个字体对象，后面绘图使用
font = FontProperties(fname="C:\\Windows\\Fonts\\SimHei.ttf")  # , size=16

#5 这里定义一个绘制分类评估器决策边界的函数
#参数：classifier 训练后的分类器； X，y：输入的特征变量和目标变量
#参数：title：图形标题
def plot_decision_boundary(classifier, X, y, title):
  xmin, xmax = np.min(X[:, 0]) - 0.05, np.max(X[:, 0]) + 0.05
  ymin, ymax = np.min(X[:, 1]) - 0.05, np.max(X[:, 1]) + 0.05

  step = 0.01
  xx, yy = np.meshgrid(np.arange(xmin, xmax, step), np.arange(ymin, ymax, step))
  Z = classifier.predict(np.hstack((xx.ravel()[:, np.newaxis], yy.ravel()[:, np.newaxis])))
  Z = Z.reshape(xx.shape)
  
  colormap = plt.cm.Paired
  plt.contourf(xx, yy, Z, cmap=colormap)
  color_map_samples = {0: (0, 0, .9), 1: (1, 0, 0), 2: (.8, .6, 0)}
  colors = [color_map_samples[c] for c in y]
  plt.scatter(X[:, 0], X[:, 1], c=colors, edgecolors="black")
  plt.xlim(xmin, xmax)
  plt.ylim(ymin, ymax)
  plt.title(title, fontproperties=font)
# end of plot_decision_boundary()


#6 初始化画布
fig = plt.figure(figsize=(10, 8))
fig.canvas.manager.set_window_title("VotingClassifier集成模型")  # Matplotlib >= 3.4
#fig.canvas.set_window_title("VotingClassifier集成模型")         # Matplotlib < 3.4

#7.1 训练一个决策树模型DecisionTreeClassifier
tree = DecisionTreeClassifier(min_samples_split=5, min_samples_leaf=3, random_state=seed)
tree.fit(X_train, y_train)
plt.subplot(2, 2, 1)
plot_decision_boundary(tree, X_train, y_train, "决策树模型决策边界")

#7.2 训练一个多层感知机模型MLPClassifier
mlp = MLPClassifier(hidden_layer_sizes=(10,), max_iter=10000, random_state=seed)
mlp.fit(X_train, y_train)
plt.subplot(2, 2, 2)
plot_decision_boundary(mlp, X_train, y_train, "多层感知模型机决策边界")

#7.3 训练一个k最近邻模型KNeighborsClassifier
knn = KNeighborsClassifier(n_neighbors=3)
knn.fit(X_train, y_train)
plt.subplot(2, 2, 3)
plot_decision_boundary(knn, X_train, y_train, "最近邻模型决策边界")

#7.4 训练一个投票分类器(集成模型)，并组合上面三个模型
voting_clf = VotingClassifier(estimators=[("Tree", tree), ("MLP", mlp), ("kNN", knn)], voting="hard")
voting_clf.fit(X_train, y_train)
plt.subplot(2, 2, 4)
plot_decision_boundary(voting_clf, X_train, y_train, "投票分类集成模型决策边界")

#8 最后展示
plt.tight_layout()
plt.show()
 